<?php
require_once 'includes/auth.php';

$msg = '';
$msgType = 'success';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $act = $_POST['action'] ?? '';

    if ($act === 'tx_update') {
        $txId = $_POST['id'];
        $status = $_POST['status']; // 'completed' or 'failed'
        $type = $_POST['type'];   // 'deposit' or 'withdrawal'

        try {
            $db->beginTransaction();
            $stmt = $db->prepare("SELECT * FROM transactions WHERE id = ? FOR UPDATE");
            $stmt->execute([$txId]);
            $tx = $stmt->fetch();

            if (!$tx || $tx['status'] !== 'pending') {
                throw new Exception("Transaction is no longer pending or does not exist.");
            }

            // Update Transaction Status
            $db->prepare("UPDATE transactions SET status = ?, updated_at = NOW() WHERE id = ?")->execute([$status, $txId]);

            // Handle Balance Refunds/Credits
            if ($status === 'failed' && $type === 'withdrawal') {
                // Refund the user's balance because their withdrawal failed/rejected
                $db->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?")->execute([$tx['amount'], $tx['user_id']]);
            } elseif ($status === 'completed' && $type === 'deposit') {
                // Determine Welcome Bonus
                $stmtCount = $db->prepare("SELECT COUNT(*) FROM transactions WHERE user_id=? AND type='deposit' AND status='completed' AND id != ?");
                $stmtCount->execute([$tx['user_id'], $txId]);
                $depCount = (int)$stmtCount->fetchColumn(); 

                $sStmt = $db->query("SELECT k, v FROM settings WHERE k IN ('bonus_1st_dep', 'bonus_2nd_dep', 'bonus_3rd_dep')");
                $settings = $sStmt->fetchAll(PDO::FETCH_KEY_PAIR);
                $bPct = 0;
                if ($depCount === 0) $bPct = (float)($settings['bonus_1st_dep'] ?? 0);
                elseif ($depCount === 1) $bPct = (float)($settings['bonus_2nd_dep'] ?? 0);
                elseif ($depCount === 2) $bPct = (float)($settings['bonus_3rd_dep'] ?? 0);
                
                $bonusAmt = $bPct > 0 ? ($tx['amount'] * ($bPct / 100)) : 0;
                $totalCredit = $tx['amount'] + $bonusAmt;

                // Credit the user's balance because their manual deposit was approved
                $db->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?")->execute([$totalCredit, $tx['user_id']]);

                // Also log bonus as a separate transaction for clarity
                if ($bonusAmt > 0) {
                    $bTxId = 'BONUS_' . time() . '_' . random_int(1000, 9999);
                    $db->prepare("INSERT INTO transactions (user_id, amount, type, status, payout_details, transaction_id, created_at, updated_at) VALUES (?, ?, 'deposit', 'completed', ?, ?, NOW(), NOW())")
                       ->execute([$tx['user_id'], $bonusAmt, "Welcome Bonus ($bPct%)", $bTxId]);
                }
            }

            $db->commit();
            
            $actionWord = $status === 'completed' ? 'Approved' : 'Rejected';
            $msg = "Successfully $actionWord " . h($type) . " of ₹" . money($tx['amount']) . " for User: " . h($tx['user_id']);
            $msgType = 'success';

        } catch (Exception $e) {
            if ($db->inTransaction()) $db->rollBack();
            $msg = $e->getMessage();
            $msgType = 'error';
        }
    }
}

// Fetch Pending
$pendingTxs = $db->query("SELECT * FROM transactions WHERE status = 'pending' ORDER BY created_at ASC")->fetchAll();

// Fetch Recent History
$history = $db->query("SELECT * FROM transactions WHERE status != 'pending' ORDER BY updated_at DESC LIMIT 50")->fetchAll();

require_once 'includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-2xl font-bold text-white tracking-tight">Finance Cashier</h1>
    <p class="text-sm text-slate-400 mt-1">Review, approve, or reject user deposits and withdrawals.</p>
</div>

<?php if($msg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border <?= $msgType === 'success' ? 'bg-green-500/10 border-green-500/20 text-green-400' : 'bg-red-500/10 border-red-500/20 text-red-400' ?>">
        <i data-feather="<?= $msgType === 'success' ? 'check-circle' : 'alert-circle' ?>" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($msg) ?></p>
    </div>
<?php endif; ?>


<!-- Pending Requests Section -->
<div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 mb-8 overflow-hidden">
    <div class="p-6 border-b border-slate-700/50 flex items-center justify-between">
        <h2 class="text-lg font-bold text-white flex items-center gap-2">
            <i data-feather="clock" class="w-5 h-5 text-yellow-400"></i> Pending Requests
            <?php if(count($pendingTxs) > 0): ?>
                <span class="bg-yellow-500/20 text-yellow-400 border border-yellow-500/30 text-xs px-2 py-0.5 rounded-full ml-2"><?= count($pendingTxs) ?></span>
            <?php endif; ?>
        </h2>
    </div>

    <?php if(empty($pendingTxs)): ?>
        <div class="p-12 text-center text-slate-500 font-medium flex flex-col items-center">
            <div class="w-16 h-16 bg-slate-800 rounded-full flex items-center justify-center mb-4">
                <i data-feather="check" class="w-8 h-8 text-slate-600"></i>
            </div>
            All caught up! No pending financial requests.
        </div>
    <?php else: ?>
        <div class="overflow-x-auto">
            <table class="w-full text-left border-collapse">
                <thead>
                    <tr class="bg-slate-900/50 border-b border-slate-700/50">
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Date / Time</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Type</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">User ID</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Amount</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Gateway Ref. / Notes</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-right">Decision</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-slate-800/50">
                    <?php foreach ($pendingTxs as $tx): ?>
                    <tr class="hover:bg-slate-800/40 transition-colors">
                        <td class="p-4 text-sm text-slate-400"><?= date('M d, Y H:i', strtotime($tx['created_at'])) ?></td>
                        <td class="p-4">
                            <?php if ($tx['type'] === 'deposit'): ?>
                                <span class="bg-green-500/10 text-green-400 border border-green-500/20 px-2.5 py-1 rounded text-[10px] font-bold uppercase tracking-wider">Deposit</span>
                            <?php else: ?>
                                <span class="bg-purple-500/10 text-purple-400 border border-purple-500/20 px-2.5 py-1 rounded text-[10px] font-bold uppercase tracking-wider">Withdrawal</span>
                            <?php endif; ?>
                        </td>
                        <td class="p-4 font-bold text-slate-200"><?= h($tx['user_id']) ?></td>
                        <td class="p-4 text-lg font-black tracking-tight <?= $tx['type'] === 'deposit' ? 'text-green-400' : 'text-slate-100' ?>">
                            ₹<?= money($tx['amount']) ?>
                        </td>
                        <td class="p-4 text-xs text-slate-400 max-w-[200px] truncate" title="<?= h($tx['payout_details'] ?? $tx['transaction_id']) ?>">
                            <?= h($tx['payout_details'] ?? $tx['transaction_id']) ?>
                        </td>
                        <td class="p-4 text-right">
                            <form method="POST" class="flex items-center justify-end gap-2">
                                <input type="hidden" name="action" value="tx_update">
                                <input type="hidden" name="id" value="<?= $tx['id'] ?>">
                                <input type="hidden" name="type" value="<?= $tx['type'] ?>">
                                
                                <button name="status" value="completed" class="bg-green-600 hover:bg-green-500 text-white w-9 h-9 flex items-center justify-center rounded-lg shadow-lg shadow-green-500/20 transition-all border border-green-500/30" title="Approve Request">
                                    <i data-feather="check" class="w-5 h-5"></i>
                                </button>
                                <button name="status" value="failed" class="bg-red-600 hover:bg-red-500 text-white w-9 h-9 flex items-center justify-center rounded-lg shadow-lg shadow-red-500/20 transition-all border border-red-500/30" title="Reject Request">
                                    <i data-feather="x" class="w-5 h-5"></i>
                                </button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<!-- Processed History -->
<div class="bg-slate-800/50 backdrop-blur-sm border border-slate-700/50 rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden">
    <div class="p-6 border-b border-slate-700/50">
        <h2 class="text-lg font-bold text-white flex items-center gap-2">
            <i data-feather="list" class="w-5 h-5 text-slate-400"></i> Recent History
        </h2>
    </div>

    <div class="overflow-x-auto">
        <table class="w-full text-left border-collapse">
            <thead>
                <tr class="bg-slate-900/50 border-b border-slate-700/50">
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Status / Type</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Player</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Amount</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest right-align text-right">Processed At</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-slate-800/50">
                <?php if(empty($history)): ?>
                    <tr><td colspan="4" class="p-8 text-center text-slate-500">No transaction history found.</td></tr>
                <?php else: ?>
                    <?php foreach ($history as $h): ?>
                    <tr class="hover:bg-slate-800/20 transition-colors">
                        <td class="p-4">
                            <div class="flex items-center gap-2">
                                <?php if($h['status'] === 'completed'): ?>
                                    <div class="w-6 h-6 rounded-full bg-green-500/20 flex items-center justify-center text-green-400 shrink-0"><i data-feather="check" class="w-3 h-3"></i></div>
                                <?php else: ?>
                                    <div class="w-6 h-6 rounded-full bg-red-500/20 flex items-center justify-center text-red-400 shrink-0"><i data-feather="x" class="w-3 h-3"></i></div>
                                <?php endif; ?>
                                <span class="text-xs font-bold text-slate-300 uppercase tracking-wider"><?= h($h['type']) ?></span>
                            </div>
                        </td>
                        <td class="p-4 font-bold text-slate-200 text-sm"><?= h($h['user_id']) ?></td>
                        <td class="p-4 font-bold text-slate-100 text-sm">₹<?= money($h['amount']) ?></td>
                        <td class="p-4 text-xs text-slate-500 text-right"><?= date('M d, Y H:i', strtotime($h['updated_at'])) ?></td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>
