<?php
require_once 'includes/auth.php';

$msg = '';
$msgType = 'success';

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $act = $_POST['action'] ?? '';

    try {
        if ($act === 'add_fielding') {
            $userIdToAdd = trim($_POST['user_id'] ?? '');
            if (!$userIdToAdd) throw new Exception("User ID is required.");
            
            $checkUser = $db->prepare("SELECT user_id FROM users WHERE user_id = ?");
            $checkUser->execute([$userIdToAdd]);
            if ($checkUser->fetch()) {
                $stmt = $db->prepare("INSERT IGNORE INTO fielding_users (user_id, added_by) VALUES (?, ?)");
                $stmt->execute([$userIdToAdd, $_SESSION['admin_user']]);
                $msg = 'User ' . h($userIdToAdd) . ' added to fielding list. They will now lose ~90% of games.';
            } else {
                throw new Exception('User ID not found in system.');
            }
        }
        
        if ($act === 'remove_fielding') {
            $userIdToRemove = trim($_POST['user_id'] ?? '');
            if (!$userIdToRemove) throw new Exception("User ID is required.");
            
            $stmt = $db->prepare("DELETE FROM fielding_users WHERE user_id = ?");
            $stmt->execute([$userIdToRemove]);
            $msg = 'User ' . h($userIdToRemove) . ' removed from fielding list.';
        }
    } catch (Exception $e) {
        $msg = $e->getMessage();
        $msgType = 'error';
    }
}

// Get all fielding users
$fieldingUsers = $db->query("
    SELECT f.*, u.phone, u.balance 
    FROM fielding_users f 
    LEFT JOIN users u ON f.user_id COLLATE utf8mb4_general_ci = u.user_id COLLATE utf8mb4_general_ci
    ORDER BY f.created_at DESC
")->fetchAll();

require_once 'includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-2xl font-bold text-white tracking-tight flex items-center gap-2">
        <i data-feather="crosshair" class="w-6 h-6 text-red-500"></i> Fielding System
    </h1>
    <p class="text-sm text-slate-400 mt-1">Users placed in this list are algorithmically forced to lose 90% of their games across Wingo, Mines, and Crash.</p>
</div>

<?php if($msg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border <?= $msgType === 'success' ? 'bg-green-500/10 border-green-500/20 text-green-400' : 'bg-red-500/10 border-red-500/20 text-red-400' ?>">
        <i data-feather="<?= $msgType === 'success' ? 'check-circle' : 'alert-circle' ?>" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($msg) ?></p>
    </div>
<?php endif; ?>

<div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
    
    <!-- Add User Card -->
    <div class="md:col-span-2 border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 p-6 relative overflow-hidden">
        <div class="absolute -right-6 -top-6 w-32 h-32 bg-red-500/10 rounded-full blur-2xl pointer-events-none"></div>
        <h2 class="text-lg font-bold text-white flex items-center gap-2 mb-6 relative z-10">
            <i data-feather="user-plus" class="w-5 h-5 text-red-400"></i> Target a Player
        </h2>
        
        <form method="POST" class="relative z-10 flex flex-col md:flex-row gap-4 items-end">
            <div class="flex-1 w-full">
                <label class="text-xs font-bold text-slate-400 uppercase tracking-widest pl-1 mb-2 block">Player User ID</label>
                <input type="text" name="user_id" placeholder="Enter User ID (e.g. 26011156)" class="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-3 text-white placeholder:text-slate-600 focus:outline-none focus:border-red-500 transition-colors font-mono tracking-wider" required>
            </div>
            <button type="submit" name="action" value="add_fielding" class="w-full md:w-auto bg-red-600 hover:bg-red-500 text-white px-6 py-3 rounded-xl font-bold transition-all shadow-lg shadow-red-600/20 border border-red-500/50 flex items-center justify-center gap-2">
                <i data-feather="target" class="w-4 h-4"></i> Add to List
            </button>
        </form>
    </div>

    <!-- Stats Card -->
    <div class="md:col-span-1 border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 p-6 flex flex-col items-center justify-center text-center relative overflow-hidden">
        <div class="absolute inset-0 bg-gradient-to-b from-transparent to-red-900/10 pointer-events-none"></div>
        <i data-feather="users" class="w-8 h-8 text-slate-600 mb-2"></i>
        <div class="text-4xl font-black text-red-500 mb-1 leading-none tracking-tighter"><?= count($fieldingUsers) ?></div>
        <div class="text-xs font-bold text-slate-500 uppercase tracking-widest">Active Targets</div>
    </div>

</div>

<!-- List of Fielding Users -->
<div class="border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden">
    <div class="p-6 border-b border-slate-700/50 flex items-center justify-between bg-slate-800/80">
        <h2 class="text-lg font-bold text-white flex items-center gap-2">
            <i data-feather="list" class="w-5 h-5 text-slate-400"></i> Fielded Accounts
        </h2>
    </div>

    <div class="overflow-x-auto min-h-[300px]">
        <table class="w-full text-left border-collapse">
            <thead>
                <tr class="bg-slate-900/50 border-b border-slate-700/50">
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest pl-6">Player ID</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Phone</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Current Balance</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Targeted By</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Time Added</th>
                    <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-right pr-6">Release Status</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-slate-800/50">
                <?php if(empty($fieldingUsers)): ?>
                    <tr><td colspan="6" class="p-16 text-center text-slate-500 font-medium">No players are currently fielded.</td></tr>
                <?php else: ?>
                    <?php foreach($fieldingUsers as $fu): ?>
                        <tr class="hover:bg-slate-800/40 transition-colors group">
                            <td class="p-4 pl-6 font-bold text-slate-200">
                                <?= h($fu['user_id']) ?>
                            </td>
                            <td class="p-4 text-slate-400 text-sm">
                                <?= h($fu['phone'] ?? 'N/A') ?>
                            </td>
                            <td class="p-4">
                                <span class="font-black tracking-tight <?= $fu['balance'] > 0 ? 'text-green-400' : 'text-slate-500' ?>">
                                    ₹<?= money((float)($fu['balance'] ?? 0)) ?>
                                </span>
                            </td>
                            <td class="p-4">
                                <span class="bg-slate-700 text-slate-300 px-2.5 py-1 rounded text-[10px] font-bold uppercase tracking-wider">
                                    <?= h($fu['added_by'] ?? 'System') ?>
                                </span>
                            </td>
                            <td class="p-4 text-slate-500 font-mono text-xs">
                                <?= date('d M, Y H:i', strtotime($fu['created_at'])) ?>
                            </td>
                            <td class="p-4 pr-6 text-right">
                                <form method="POST" onsubmit="return confirm('Release user <?= h($fu['user_id']) ?> from fielding? They will return to normal RNG probabilities.');">
                                    <input type="hidden" name="action" value="remove_fielding">
                                    <input type="hidden" name="user_id" value="<?= h($fu['user_id']) ?>">
                                    <button type="submit" class="bg-emerald-500/10 hover:bg-emerald-500/20 text-emerald-500 font-bold border border-emerald-500/20 px-4 py-2 rounded-xl text-xs transition-colors flex items-center gap-2 justify-end ml-auto">
                                        <i data-feather="unlock" class="w-3.5 h-3.5"></i> Release
                                    </button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>
