<?php
require_once 'includes/auth.php';

$msg = '';
$msgType = 'success';

// Ensure table exists - moved to setup/install but fallback is here
$db->exec("CREATE TABLE IF NOT EXISTS gift_codes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    code VARCHAR(50) UNIQUE NOT NULL,
    amount DECIMAL(12,2) NOT NULL,
    uses_allowed INT DEFAULT 1,
    uses_count INT DEFAULT 0,
    expires_at DATETIME NULL,
    active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $act = $_POST['action'] ?? '';

    try {
        if ($act === 'create_gift') {
            $code = strtoupper(trim($_POST['code'] ?? ''));
            $amount = (float)($_POST['amount'] ?? 0);
            $uses = (int)($_POST['uses_allowed'] ?? 1);
            
            if (!$code) throw new Exception("Gift code text is required.");
            if ($amount <= 0) throw new Exception("Bonus amount must be positive.");
            if ($uses <= 0) throw new Exception("Max uses must be at least 1.");

            $stmt = $db->prepare("INSERT INTO gift_codes (code, amount, uses_allowed) VALUES (?, ?, ?)");
            $stmt->execute([$code, $amount, $uses]);
            $msg = "Gift code '$code' created! It can be claimed $uses time(s) for ₹" . money($amount);
            $msgType = 'success';
        }

        if ($act === 'toggle_gift') {
            $id = (int)$_POST['id'];
            $st = (int)$_POST['status'];
            $db->prepare("UPDATE gift_codes SET active = ? WHERE id = ?")->execute([$st, $id]);
            $msg = "Gift code status updated.";
        }
        
    } catch (Exception $e) {
        // usually duplicate entry on code
        if ($e->getCode() == 23000) {
            $msg = "Error: That gift code already exists.";
        } else {
            $msg = "Error: " . $e->getMessage();
        }
        $msgType = 'error';
    }
}

// Fetch Codes
$gifts = $db->query("SELECT * FROM gift_codes ORDER BY created_at DESC")->fetchAll();

require_once 'includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-2xl font-bold text-white tracking-tight flex items-center gap-2">
        <i data-feather="gift" class="w-6 h-6 text-pink-500"></i> Gift Code Generator
    </h1>
    <p class="text-sm text-slate-400 mt-1">Create promotional codes that add direct balances to active players' wallets.</p>
</div>

<?php if($msg): ?>
    <div class="mb-6 p-4 rounded-xl flex items-start gap-3 border <?= $msgType === 'success' ? 'bg-green-500/10 border-green-500/20 text-green-400' : 'bg-red-500/10 border-red-500/20 text-red-400' ?>">
        <i data-feather="<?= $msgType === 'success' ? 'check-circle' : 'alert-circle' ?>" class="w-5 h-5 shrink-0 mt-0.5"></i>
        <p class="text-sm font-medium"><?= h($msg) ?></p>
    </div>
<?php endif; ?>

<div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
    
    <!-- Create Form -->
    <div class="md:col-span-1 border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 p-6 relative overflow-hidden h-fit sticky top-6">
        <div class="absolute -right-12 -bottom-12 w-40 h-40 bg-pink-500/10 rounded-full blur-2xl pointer-events-none"></div>
        <h2 class="text-lg font-bold text-white flex items-center gap-2 mb-6">
            <i data-feather="plus-circle" class="w-5 h-5 text-pink-400"></i> New Code
        </h2>
        
        <form method="POST" class="flex flex-col gap-5 relative z-10">
            <input type="hidden" name="action" value="create_gift">
            
            <div>
                <label class="text-xs font-bold text-slate-400 uppercase tracking-widest pl-1 mb-2 block">Gift Code Phrase</label>
                <div class="relative">
                    <i data-feather="tag" class="w-4 h-4 absolute left-4 top-1/2 -translate-y-1/2 text-slate-500"></i>
                    <input type="text" name="code" placeholder="e.g. MEGAWIN50" class="w-full bg-slate-900 border border-slate-700 rounded-xl pl-11 pr-4 py-3 text-white placeholder:text-slate-600 focus:outline-none focus:border-pink-500 transition-colors uppercase font-mono tracking-wider font-bold" required>
                </div>
            </div>

            <div>
                <label class="text-xs font-bold text-slate-400 uppercase tracking-widest pl-1 mb-2 block">Bonus ₹ Amount</label>
                <div class="relative">
                    <span class="absolute left-4 top-1/2 -translate-y-1/2 text-slate-500 font-bold">₹</span>
                    <input type="number" step="0.1" name="amount" placeholder="e.g. 50" class="w-full bg-slate-900 border border-slate-700 rounded-xl pl-11 pr-4 py-3 text-white placeholder:text-slate-600 focus:outline-none focus:border-pink-500 transition-colors font-bold text-lg" required>
                </div>
            </div>

            <div>
                <label class="text-xs font-bold text-slate-400 uppercase tracking-widest pl-1 mb-2 block">Max Claims Allowed</label>
                <input type="number" name="uses_allowed" value="1" min="1" class="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-3 text-white placeholder:text-slate-600 focus:outline-none focus:border-pink-500 transition-colors font-bold text-center text-lg" required>
                <p class="text-[10px] text-slate-500 text-center mt-1.5 uppercase tracking-wide">Number of unique players who can use this.</p>
            </div>

            <button type="submit" class="w-full bg-pink-600 hover:bg-pink-500 text-white mt-2 py-3.5 rounded-xl font-bold transition-all shadow-lg shadow-pink-600/20 border border-pink-500/50 flex items-center justify-center gap-2">
                <i data-feather="check" class="w-4 h-4"></i> Generate Gift
            </button>
        </form>
    </div>

    <!-- Active List -->
    <div class="md:col-span-2 border border-slate-700/50 bg-slate-800/50 backdrop-blur-sm rounded-2xl shadow-lg shadow-slate-900/50 overflow-hidden flex flex-col">
        <div class="p-6 border-b border-slate-700/50 flex items-center justify-between bg-slate-800/80">
            <h2 class="text-lg font-bold text-white flex items-center gap-2">
                <i data-feather="list" class="w-5 h-5 text-slate-400"></i> Code Registry
            </h2>
        </div>
        
        <div class="overflow-x-auto min-h-[400px]">
            <table class="w-full text-left border-collapse">
                <thead>
                    <tr class="bg-slate-900/50 border-b border-slate-700/50">
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest pl-6">Code Hash</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest">Reward</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-center">Claims</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-center">State</th>
                        <th class="p-4 text-xs font-bold text-slate-400 uppercase tracking-widest text-right pr-6">Toggle</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-slate-800/50 text-sm">
                    <?php if(empty($gifts)): ?>
                        <tr><td colspan="5" class="p-16 text-center text-slate-500 font-medium">No gift codes generated yet.</td></tr>
                    <?php else: ?>
                        <?php foreach($gifts as $g): 
                            $isDepleted = $g['uses_count'] >= $g['uses_allowed'];
                            $isActive = $g['active'] == 1;
                        ?>
                            <tr class="hover:bg-slate-800/40 transition-colors group <?= $isDepleted ? 'opacity-60' : '' ?>">
                                <td class="p-4 pl-6 pl-6">
                                    <div class="font-black font-mono text-pink-400 tracking-wider flex items-center gap-2">
                                        <?= h($g['code']) ?>
                                    </div>
                                    <div class="text-[10px] text-slate-500 font-medium mt-1 uppercase tracking-widest">
                                        <?= date('M d, y H:i', strtotime($g['created_at'])) ?>
                                    </div>
                                </td>
                                <td class="p-4">
                                    <span class="font-black text-white text-base">₹<?= money($g['amount']) ?></span>
                                </td>
                                <td class="p-4 text-center">
                                    <span class="bg-slate-900 text-slate-300 font-bold border border-slate-700 rounded-lg px-3 py-1 text-xs whitespace-nowrap">
                                        <b class="<?= $isDepleted ? 'text-red-400' : 'text-green-400' ?>"><?= $g['uses_count'] ?></b> / <?= $g['uses_allowed'] ?>
                                    </span>
                                </td>
                                <td class="p-4 text-center">
                                    <?php if ($isActive && !$isDepleted): ?>
                                        <div class="inline-flex items-center gap-1.5 bg-green-500/10 text-green-400 border border-green-500/20 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">
                                            <div class="w-1.5 h-1.5 rounded-full bg-green-400 animate-pulse"></div> Live
                                        </div>
                                    <?php elseif ($isDepleted): ?>
                                        <div class="inline-flex items-center gap-1.5 bg-red-500/10 text-red-400 border border-red-500/20 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">
                                            <div class="w-1.5 h-1.5 rounded-full bg-red-500"></div> Depleted
                                        </div>
                                    <?php else: ?>
                                        <div class="inline-flex items-center gap-1.5 bg-slate-500/10 text-slate-400 border border-slate-500/30 px-2 py-0.5 rounded text-[10px] font-bold uppercase tracking-wider">
                                            <div class="w-1.5 h-1.5 rounded-full bg-slate-500"></div> Disabled
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td class="p-4 pr-6 text-right">
                                    <?php if(!$isDepleted): ?>
                                        <form method="POST">
                                            <input type="hidden" name="action" value="toggle_gift">
                                            <input type="hidden" name="id" value="<?= $g['id'] ?>">
                                            <input type="hidden" name="status" value="<?= $isActive ? '0' : '1' ?>">
                                            <?php if ($isActive): ?>
                                                <button type="submit" class="bg-slate-700 hover:bg-slate-600 text-slate-300 border border-slate-600 px-3 py-1.5 rounded-lg text-xs font-bold transition-all ml-auto">Deactivate</button>
                                            <?php else: ?>
                                                <button type="submit" class="bg-blue-600/20 hover:bg-blue-600/30 text-blue-400 border border-blue-500/30 px-3 py-1.5 rounded-lg text-xs font-bold transition-all ml-auto">Activate</button>
                                            <?php endif; ?>
                                        </form>
                                    <?php else: ?>
                                        <span class="text-slate-600 font-bold text-xs uppercase tracking-wider">Empty</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

</div>

<?php require_once 'includes/footer.php'; ?>
